﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using SATO.MLComponent;

/// <summary>
/// Multi LABELIST Component サンプルプログラム(ステータスL)
/// 
/// [概要]
/// タフアーム(LR4NX-FA)とステータスLを使用してラベル発行を行うサンプルプログラムです。
/// プリンタコマンド(SBPL)やステータスLについての詳細は、LR4NX-FAプログラミングリファレンスを参照してください。
/// 
/// [開発環境]
/// ・タフアーム LR4NX-FA 305dpi (I/F：LAN、通信プロトコル：STATUSL)
/// ・Multi LABELIST Component
/// 
/// [ご注意]
/// ・本サンプルプログラムの内容は訂正・改善のため予告なく変更されることがあります。
/// ・本サンプルプログラムを運用した結果の影響については一切責任を負いかねますのでご了承ください。
/// 
/// ※2020/04/20 コードの一部誤記を修正しました。
/// </summary>
namespace MLComponent_StatusL_CS
{
    public partial class Form1 : Form
    {
        private MLComponent MLComponent;

        private const string STX = "\x02";
        private const string ETX = "\x03";
        private const string ACK = "\x06";
        private const string ESC = "\x1B";

        private bool blnStop;

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            //MLComponent初期化
            MLComponent = new MLComponent();

            txtIPAddress.Text = "0.0.0.0";
            txtPortNo.Text = "9100";

            //レイアウトファイルとデータファイル
            txtLayoutFile.Text = Application.StartupPath + "\\LR4NX-FA_Sample.mllayx";
            txtDataFile.Text = Application.StartupPath + "\\LR4NX-FA_Sample.txt";

            cmbDataType.Items.Add("タブ区切り");
            cmbDataType.Items.Add("カンマ区切り");
            cmbDataType.Items.Add("スペース区切り");
            cmbDataType.SelectedIndex = 0;

            lblStatus.Text = "";
            lblItemCount.Text = "";
            btnStop.Enabled = false;
        }

        /// <summary>
        /// ラベル発行を行います。
        /// </summary>
        private void btnPrint_Click(object sender, EventArgs e)
        {
            int Result = 0;
            string ReturnStatus = "";          //返送ステータス
            string StatusCode = "";            //ステータス文字
            string CompletionQuantity = "";    //貼付完了枚数
            bool PrinterSendable = false;
            int index = 0;

            //通信プロトコル(ステータスLでは「Status3」を選択)
            MLComponent.Protocol = SATO.MLComponent.Protocols.Status3;
            //通信タイムアウト値
            MLComponent.Timeout = 3;
            //レイアウトファイル
            try
            {
                MLComponent.LayoutFile = txtLayoutFile.Text;
            }catch (Exception ex) {
                MessageBox.Show("レイアウトファイルの指定が不正です。" + System.Environment.NewLine + ex.Message);
                return;
            }
            //データタイプ
            MLComponent.PrnDataType = (PrnDataTypes)cmbDataType.SelectedIndex;


            //プリンタに接続(ポートオープン)
            if (ConnectPrinter() == false)
            {
                lblStatus.Text = "プリンタ接続エラー";
                return;
            }

            blnStop = false;
            btnStop.Enabled = true;
            btnPrint.Enabled = false;
            btnSendCancel.Enabled = false;
            lblItemCount.Text = "";

            //プリンタステータス確認(発行前)
            PrinterSendable = false;
            while(PrinterSendable == false)
            {
                Application.DoEvents();

                //中断ボタンの判定
                if (blnStop == true)
                {
                    if (MessageBox.Show("発行を中止しますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                    {
                        break;
                    }
                    else
                    {
                        blnStop = false;
                    }
                }

                //プリンタステータス取得
                Result = MLComponent.GetStatus(ref ReturnStatus);
                if (Result != 0)
                {
                    MessageBox.Show("プリンタステータス取得時にエラーが発生しました。発行を中止します。 (エラーコード : " + Result + ")");
                    break;
                }

                StatusCode = ReturnStatus.Substring(2, 1);  //プリンタステータス
                lblStatus.Text = ReturnStatus + System.Environment.NewLine + "(" + StatusLMessage(StatusCode) + ")";

                //プリンタ送信可能チェック
                switch (StatusCode)
                {
                    case "A":
                    case "B":
                        //送信可能
                        PrinterSendable = true;
                        break;
                    default:
                        //待機
                        PrinterSendable = false;
                        break;
                }
            }
            
            //中断・プリンタ状態確認失敗時の処理
            if (PrinterSendable == false)
            {
                MLComponent.ClosePort();
                btnStop.Enabled = false;
                btnPrint.Enabled = true;
                btnSendCancel.Enabled = true;
                return;
            }

            //データファイル読み込み
            IEnumerable<string> lines = File.ReadLines(txtDataFile.Text);
            index = 0;
            lblItemCount.Text = index.ToString() + " / " + lines.Count().ToString();

            foreach (string line in lines)
            {
                //パス指定コマンド<PA>送信（パス回数を"000001"にセットしてワーク１回分貼付を行わない）
                if ((index == 0) && (chkPassThrough.Checked == true) && (SetPathThrough() == false))
                {
                    //コマンド送信失敗時の処理
                    lblStatus.Text = "パス指定コマンド送信エラー";
                    break;
                }

                //貼付完了枚数指定コマンド<ES>送信（枚数を"000000"にセットしてプリンタの貼付完了枚数をクリア）
                if ((chkCheckQTY.Checked == true) && (SetCompletionQty() == false))
                {
                    //コマンド送信失敗時の処理
                    lblStatus.Text = "貼付完了枚数指定コマンド送信エラー";
                    break;
                }

                //印字データセット
                MLComponent.PrnData = line;

                //印字コマンド送信
                Result = MLComponent.Output();

                if (Result != 0)
                {
                    MessageBox.Show("プリンタへの印字コマンド送信時にエラーが発生しました。発行を中止します。(エラーコード : " + Result + ")");
                    break;
                }

                //プリンタステータス確認（発行後）
                PrinterSendable = false;
                while (PrinterSendable == false)
                {
                    Application.DoEvents();

                    //中断ボタンの判定
                    if (blnStop == true)
                    {
                        if (MessageBox.Show("発行を中止しますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                        {
                            break;
                        }
                        else
                        {
                            blnStop = false;
                        }
                    }

                    //プリンタステータス受信
                    Result = MLComponent.GetStatus(ref ReturnStatus);
                    if (Result != 0)
                    {
                        MessageBox.Show("プリンタステータス取得時にエラーが発生しました。発行を中止します。(エラーコード : " + Result + ")");
                        break;
                    }

                    StatusCode = ReturnStatus.Substring(2, 1);          //プリンタステータス
                    CompletionQuantity = ReturnStatus.Substring(3, 6);  //貼付完了枚数
                    lblStatus.Text = ReturnStatus + System.Environment.NewLine + "(" + StatusLMessage(StatusCode) + ")";

                    //貼付完了チェック
                    switch (StatusCode)
                    {
                        case "A":
                        case "B":
                            PrinterSendable = true;
                            //<ES>使用時：ステータスLの貼付完了枚数がカウントアップするまで待機
                            if ((chkCheckQTY.Checked == true) && (int.Parse(CompletionQuantity) < 1))
                            {
                                PrinterSendable = false;
                            }
                            break;
                        default:
                            //待機
                            PrinterSendable = false;
                            break;
                    }
                }

                //発行が中止されたらプリンタのバッファデータをキャンセル
                if (PrinterSendable == false)
                {
                    if (MessageBox.Show("プリンタのバッファデータをキャンセルしますか？",
                        "キャンセル確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                    {
                        //キャンセルコマンド送信
                        Result = MLComponent.SendCancel();
                        if (Result != 0)
                        {
                            MessageBox.Show("キャンセルコマンド送信時にエラーが発生しました。(エラーコード : " + Result + ")");
                        }
                    }
                    break;
                }
                else
                {
                    index++;
                    lblItemCount.Text = index.ToString() + " / " + lines.Count().ToString();
                }
            }

            //切断
            MLComponent.ClosePort();

            lblStatus.Text = "発行完了";
            btnStop.Enabled = false;
            btnPrint.Enabled = true;
            btnSendCancel.Enabled = true;
        }

        /// <summary>
        /// プリンタに接続します。
        /// </summary>
        /// <returns>プリンタ接続結果</returns>
        private bool ConnectPrinter()
        {
            int Result;

            //プリンタのIPアドレスとポート番号
            MLComponent.Setting = "LAN:" + txtIPAddress.Text + "," + txtPortNo.Text;

            //プリンタに接続
            Result = MLComponent.OpenPort(1);
            if (Result != 0)
            {
                MessageBox.Show("プリンタ接続時にエラーが発生しました。(エラーコード : " + Result + ")");
                return false;
            }
            return true;
        }

        /// <summary>
        /// プリンタにパス指定コマンド(ESC+PA)を送信します。
        /// </summary>
        /// <returns>コマンド送信結果</returns>
        private bool SetPathThrough()
        {
            string Command;

            Command = STX + ESC + "A" + ESC + "PA000001" + ESC + "Z" + ETX;

            try
            {
                //<PA>コマンドを送信
                MLComponent.SendStringData(0, Command, 0, ACK.ToString());
            } catch (Exception ex)
            {
                MessageBox.Show("パス指定コマンド送信時にエラーが発生しました。" + System.Environment.NewLine + ex.Message);
                return false;
            }
            return true;
        }

        /// <summary>
        /// プリンタに貼付完了枚数指定コマンド(ESC+ES)を送信します。
        /// </summary>
        /// <returns>コマンド送信結果</returns>
        private bool SetCompletionQty()
        {
            string Command;

            Command = STX + ESC + "A" + ESC + "ES000000" + ESC + "Z" + ETX;

            try
            {
                //<ES>コマンドを送信
                MLComponent.SendStringData(0, Command, 0, ACK.ToString());
            }
            catch (Exception ex)
            {
                MessageBox.Show("貼付完了枚数指定コマンド送信時にエラーが発生しました。" + System.Environment.NewLine + ex.Message);
                return false;
            }
            return true;
        }

        /// <summary>
        /// ステータスコードから内容を取得します。
        /// </summary>
        /// <param name="StatusChar">ステータスコード</param>
        /// <returns>ステータス内容</returns>
        private string StatusLMessage(string StatusChar)
        {
            string StatusString = "";

            //※ e,f,#,$,%,+ を判別するには、LR4NX-FA 本体で「ステータスL」を「詳細」に設定します
            switch (StatusChar)
            {
                case "*":    //電源投入直後の初回のみ返送（ステータスL専用）
                    StatusString = "イニシャル状態";
                    break;
                case "0":
                    StatusString = "オフライン - エラーなし";
                    break;
                case "1":
                    StatusString = "オフライン - リボン/ラベルニアエンド";
                    break;
                case "2":
                    StatusString = "オフライン - バッファニアフル";
                    break;
                case "3":
                    StatusString = "オフライン - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "A":
                    StatusString = "受信待ち - エラーなし";
                    break;
                case "B":
                    StatusString = "受信待ち - リボン/ラベルニアエンド";
                    break;
                case "C":
                    StatusString = "受信待ち - バッファニアフル";
                    break;
                case "D":
                    StatusString = "受信待ち - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "M":
                    StatusString = "印字待ち/通過待ち - エラーなし";
                    break;
                case "N":
                    StatusString = "印字待ち/通過待ち - リボン/ラベルニアエンド";
                    break;
                case "O":
                    StatusString = "印字待ち/通過待ち - バッファニアフル";
                    break;
                case "P":
                    StatusString = "印字待ち/通過待ち - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "G":
                    StatusString = "印字中 - エラーなし";
                    break;
                case "H":
                    StatusString = "印字中 - リボン/ラベルニアエンド";
                    break;
                case "I":
                    StatusString = "印字中 - バッファニアフル";
                    break;
                case "J":
                    StatusString = "印字中 - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "o":
                    StatusString = "印字吸着起動信号待ち/貼付中 - エラーなし";
                    break;
                case "p":
                    StatusString = "印字吸着起動信号待ち/貼付中 - リボン/ラベルニアエンド";
                    break;
                case "q":
                    StatusString = "印字吸着起動信号待ち/貼付中 - バッファニアフル";
                    break;
                case "r":
                    StatusString = "印字吸着起動信号待ち/貼付中 - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "s":
                    StatusString = "通過中 - エラーなし";
                    break;
                case "t":
                    StatusString = "通過中 - リボン/ラベルニアエンド";
                    break;
                case "u":
                    StatusString = "通過中 - バッファニアフル";
                    break;
                case "v":
                    StatusString = "通過中 - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "S":
                    StatusString = "解析・編集中 - エラーなし";
                    break;
                case "T":
                    StatusString = "解析・編集中 - リボン/ラベルニアエンド";
                    break;
                case "U":
                    StatusString = "解析・編集中 - バッファニアフル";
                    break;
                case "V":
                    StatusString = "解析・編集中 - リボン/ラベルニアエンド＆バッファニアフル";
                    break;
                case "a":
                    StatusString = "エラー - バッファオーバー";
                    break;
                case "b":
                    StatusString = "エラー - ヘッドオープン";
                    break;
                case "c":
                    StatusString = "エラー - ペーパーエンド/センサエラー/用紙詰まりエラー";
                    break;
                case "d":
                    StatusString = "エラー - リボンエンド";
                    break;
                case "e":
                    StatusString = "エラー - 印字飛びエラー/印字領域オーバー";
                    break;
                case "f":
                    StatusString = "エラー - センサエラー/用紙詰まりエラー";
                    break;
                case "g":
                    StatusString = "エラー - ヘッドエラー";
                    break;
                case "i":
                    StatusString = "エラー - カードエラー";
                    break;
                case "k":
                    StatusString = "エラー - その他のエラー";
                    break;
                case "l":
                    StatusString = "エラー - ラベル貼付エラー";
                    break;
                case "#":
                    StatusString = "エラー - ラベル吸着エラー";
                    break;
                case "$":
                    StatusString = "エラー - 能力オーバー";
                    break;
                case "%":
                    StatusString = "エラー - シリンダ動作不良";
                    break;
                case "+":
                    StatusString = "エラー - ユーザー専用のエラー";
                    break;
                default:
                    StatusString = StatusChar;
                    break;
            }
            return StatusString;
        }

        private void btnSelectLayout_Click(object sender, EventArgs e)
        {
            openFileDialog1.Title = "レイアウトファイルを選択";
            openFileDialog1.Filter = "レイアウトファイル (*.mllayx)|*.mllayx";
            openFileDialog1.FileName = "";

            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                txtLayoutFile.Text = openFileDialog1.FileName;
            }
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            blnStop = true;
        }

        /// <summary>
        /// プリンタにキャンセルコマンドを送り、受信バッファを取り消します。
        /// プリンタエラー等で発行を中止した場合は、キャンセルコマンドをプリンタに送信してください。
        /// </summary>
        private void btnSendCancel_Click(object sender, EventArgs e)
        {
            int Result;

            //キャンセルコマンド送信
            if (ConnectPrinter() == true)
            {
                Result = MLComponent.SendCancel();
                if (Result == 0)
                {
                    lblStatus.Text = "キャンセル送信成功";
                }
                else
                {
                    MessageBox.Show("キャンセルコマンド送信時にエラーが発生しました。(エラーコード : " + Result + ")");
                }
                MLComponent.ClosePort();
            }
        }

        private void btnSelectDataFile_Click(object sender, EventArgs e)
        {
            openFileDialog1.Title = "データファイルを選択";
            openFileDialog1.Filter = "テキスト(タブ区切り) (*.txt)|*.txt|CSV(カンマ区切り) (*.csv)|*.csv|テキスト(スペース区切り) (*.prn)|*.prn|すべてのファイル (*.*)|*.*";
            openFileDialog1.FileName = "";

            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                txtDataFile.Text = openFileDialog1.FileName;
            }
        }
    }
}
