﻿Imports System.IO

''' <summary>
''' Multi LABELIST Component サンプルプログラム(ステータスL)
''' 
''' [概要]
''' タフアーム(LR4NX-FA)とステータスLを使用してラベル発行を行うサンプルプログラムです。
''' プリンタコマンド(SBPL)やステータスLについての詳細は、LR4NX-FAプログラミングリファレンスを参照してください。
''' 
''' [開発環境]
''' ・タフアーム LR4NX-FA 305dpi (I/F：LAN、通信プロトコル：STATUSL)
''' ・Multi LABELIST Component
''' 
''' [ご注意]
''' ・本サンプルプログラムの内容は訂正・改善のため予告なく変更されることがあります。
''' ・本サンプルプログラムを運用した結果の影響については一切責任を負いかねますのでご了承ください。
''' 
''' ※2020/04/20 コードの一部誤記を修正しました。
''' </summary>
Public Class Form1

    Private MLComponent As New SATO.MLComponent.MLComponent

    Private Const STX As String = ChrW(&H2)
    Private Const ETX As String = ChrW(&H3)
    Private Const ACK As String = ChrW(&H6)
    Private Const ESC As String = ChrW(&H1B)

    Private blnStop As Boolean


    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load

        txtIPAddress.Text = "0.0.0.0"
        txtPortNo.Text = "9100"

        'レイアウトファイルとデータファイル
        txtLayoutFile.Text = Application.StartupPath & "\LR4NX-FA_Sample.mllayx"
        txtDataFile.Text = Application.StartupPath & "\LR4NX-FA_Sample.txt"

        cmbDataType.Items.Add("タブ区切り")
        cmbDataType.Items.Add("カンマ区切り")
        cmbDataType.Items.Add("スペース区切り")
        cmbDataType.SelectedIndex = 0

        lblStatus.Text = ""
        lblItemCount.Text = ""
        btnStop.Enabled = False

    End Sub

    ''' <summary>
    ''' ラベル発行を行います。
    ''' </summary>
    Private Sub btnPrint_Click(sender As Object, e As EventArgs) Handles btnPrint.Click

        Dim Result As Integer
        Dim ReturnStatus As String          '返送ステータス
        Dim StatusCode As String            'ステータス文字
        Dim CompletionQuantity As String    '貼付完了枚数
        Dim PrinterSendable As Boolean
        Dim index As Integer

        '通信プロトコル(ステータスLでは「Status3」を選択)
        MLComponent.Protocol = SATO.MLComponent.Protocols.Status3
        '通信タイムアウト値
        MLComponent.Timeout = 3
        'レイアウトファイル
        Try
            MLComponent.LayoutFile = txtLayoutFile.Text
        Catch ex As Exception
            MessageBox.Show("レイアウトファイルの指定が不正です。" & vbCrLf & ex.Message)
            Exit Sub
        End Try
        'データタイプ
        MLComponent.PrnDataType = CType(cmbDataType.SelectedIndex, SATO.MLComponent.PrnDataTypes)

        'プリンタに接続(ポートオープン)
        If ConnectPrinter() = False Then
            lblStatus.Text = "プリンタ接続エラー"
            Exit Sub
        End If

        blnStop = False
        btnStop.Enabled = True
        btnPrint.Enabled = False
        btnSendCancel.Enabled = False
        lblItemCount.Text = ""

        'プリンタステータス確認(発行前)
        PrinterSendable = False
        Do While PrinterSendable = False
            Application.DoEvents()

            '中断ボタンの判定
            If blnStop = True Then
                If MessageBox.Show("発行を中止しますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) = vbYes Then
                    Exit Do
                Else
                    blnStop = False
                End If
            End If

            'プリンタステータス受信
            Result = MLComponent.GetStatus(ReturnStatus)
            If Result <> 0 Then
                MessageBox.Show("プリンタステータス取得時にエラーが発生しました。発行を中止します。 (エラーコード : " & Result & ")")
                Exit Do
            End If

            StatusCode = ReturnStatus.Substring(2, 1)   'プリンタステータス
            lblStatus.Text = ReturnStatus & vbCrLf & "(" & StatusLMessage(StatusCode) & ")"

            'プリンタ送信可能チェック
            Select Case StatusCode
                Case "A", "B"
                    '送信可能
                    PrinterSendable = True
                Case Else
                    '待機
                    PrinterSendable = False
            End Select
        Loop

        '中断/プリンタ状態確認失敗時の処理
        If PrinterSendable = False Then
            MLComponent.ClosePort()
            btnStop.Enabled = False
            btnPrint.Enabled = True
            btnSendCancel.Enabled = True
            Exit Sub
        End If


        'データファイル読み込み
        Dim lines As IEnumerable(Of String) = File.ReadLines(txtDataFile.Text)
        index = 0
        lblItemCount.Text = index.ToString & " / " & lines.Count.ToString

        For Each line In lines

            'パス指定コマンド<PA>送信（パス回数を"000001"にセットしてワーク１回分貼付を行わない）
            If (index = 0) AndAlso (chkPassThrough.Checked = True) AndAlso (SetPathThrough() = False) Then
                'コマンド送信失敗時の処理
                lblStatus.Text = "パス指定コマンド送信エラー"
                Exit For
            End If

            '貼付完了枚数指定コマンド<ES>送信（枚数を"000000"にセットしてプリンタの貼付完了枚数をクリア）
            If (chkCheckQTY.Checked = True) AndAlso (SetCompletionQty() = False) Then
                'コマンド送信失敗時の処理
                lblStatus.Text = "貼付完了枚数指定コマンド送信エラー"
                Exit For
            End If

            '印字データセット
            MLComponent.PrnData = line

            '印字コマンド送信
            Result = MLComponent.Output()
            If Result <> 0 Then
                MessageBox.Show("プリンタへの印字コマンド送信時にエラーが発生しました。発行を中止します。(エラーコード : " & Result & ")")
                Exit For
            End If

            'プリンタステータス確認（発行後）
            PrinterSendable = False
            Do While PrinterSendable = False
                Application.DoEvents()

                '中断ボタンの判定
                If blnStop = True Then
                    If MessageBox.Show("発行を中止しますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) = vbYes Then
                        Exit Do
                    Else
                        blnStop = False
                    End If
                End If

                'プリンタステータス受信
                Result = MLComponent.GetStatus(ReturnStatus)
                If Result <> 0 Then
                    MessageBox.Show("プリンタステータス取得時にエラーが発生しました。発行を中止します。(エラーコード : " & Result & ")")
                    Exit Do
                End If

                StatusCode = ReturnStatus.Substring(2, 1)           'プリンタステータス
                CompletionQuantity = ReturnStatus.Substring(3, 6)   '貼付完了枚数
                lblStatus.Text = ReturnStatus & vbCrLf & "(" & StatusLMessage(StatusCode) & ")"

                '貼付完了チェック
                Select Case StatusCode
                    Case "A", "B"
                        PrinterSendable = True
                        '<ES>送信時：ステータスLの貼付完了枚数カウントアップを確認
                        If (chkCheckQTY.Checked = True) AndAlso (CInt(CompletionQuantity) < 1) Then
                            PrinterSendable = False
                        End If
                    Case Else
                        '待機
                        PrinterSendable = False
                End Select

            Loop

            '発行が中止されたらプリンタのバッファデータをキャンセル
            If PrinterSendable = False Then

                If MessageBox.Show("プリンタのバッファデータをキャンセルしますか？",
                               "キャンセル確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) = vbYes Then
                    'キャンセルコマンド送信
                    Result = MLComponent.SendCancel()
                    If Result <> 0 Then
                        MessageBox.Show("キャンセルコマンド送信時にエラーが発生しました。(エラーコード : " & Result & ")")
                    End If
                End If
                Exit For
            Else
                index += 1
                lblItemCount.Text = index.ToString & " / " & lines.Count.ToString
            End If
        Next

        '切断
        MLComponent.ClosePort()

        lblStatus.Text = "発行完了"
        btnStop.Enabled = False
        btnPrint.Enabled = True
        btnSendCancel.Enabled = True

    End Sub

    ''' <summary>
    ''' プリンタに接続します。
    ''' </summary>
    ''' <returns>プリンタ接続結果</returns>
    Private Function ConnectPrinter() As Boolean

        Dim Result As Integer

        'プリンタのIPアドレスとポート番号
        MLComponent.Setting = "LAN:" & txtIPAddress.Text & "," & txtPortNo.Text

        'プリンタに接続
        Result = MLComponent.OpenPort(1)

        If Result <> 0 Then
            MessageBox.Show("プリンタ接続時にエラーが発生しました。(エラーコード : " & Result & ")")
            Return False
        End If

        Return True

    End Function

    ''' <summary>
    ''' プリンタにパス指定コマンド(ESC+PA)を送信します。
    ''' </summary>
    ''' <returns>コマンド送信結果</returns>
    Private Function SetPathThrough() As Boolean

        Dim Command As String

        Command = STX & ESC & "A" & ESC & "PA000001" & ESC & "Z" & ETX

        Try
            '<PA>コマンドを送信
            MLComponent.SendStringData(0, Command, 0, ACK)
        Catch ex As Exception
            MessageBox.Show("パス指定コマンド送信時にエラーが発生しました。" & vbCrLf & ex.Message)
            Return False
        End Try

        Return True

    End Function

    ''' <summary>
    ''' プリンタに貼付完了枚数指定コマンド(ESC+ES)を送信します。
    ''' </summary>
    ''' <returns>コマンド送信結果</returns>
    Private Function SetCompletionQty() As Boolean

        Dim Command As String

        Command = STX & ESC & "A" & ESC & "ES000000" & ESC & "Z" & ETX

        Try
            '<ES>コマンドを送信
            MLComponent.SendStringData(0, Command, 0, ACK)
        Catch ex As Exception
            MessageBox.Show("貼付完了枚数指定コマンド送信時にエラーが発生しました。" & vbCrLf & ex.Message)
            Return False
        End Try

        Return True

    End Function

    ''' <summary>
    ''' ステータスコードから内容を取得します。
    ''' </summary>
    ''' <param name="StatusChar">ステータスコード</param>
    ''' <returns>ステータス内容</returns>
    Private Function StatusLMessage(ByVal StatusChar As String) As String

        Dim StatusString As String = ""

        '※ e,f,#,$,%,+ を判別するには、LR4NX-FA 本体で「ステータスL」を「詳細」に設定します
        Select Case StatusChar
            Case "*"    '電源投入直後の初回のみ返送（ステータスL専用）
                StatusString = "イニシャル状態"
            Case "0"
                StatusString = "オフライン - エラーなし"
            Case "1"
                StatusString = "オフライン - リボン/ラベルニアエンド"
            Case "2"
                StatusString = "オフライン - バッファニアフル"
            Case "3"
                StatusString = "オフライン - リボン/ラベルニアエンド＆バッファニアフル"
            Case "A"
                StatusString = "受信待ち - エラーなし"
            Case "B"
                StatusString = "受信待ち - リボン/ラベルニアエンド"
            Case "C"
                StatusString = "受信待ち - バッファニアフル"
            Case "D"
                StatusString = "受信待ち - リボン/ラベルニアエンド＆バッファニアフル"
            Case "M"
                StatusString = "印字待ち/通過待ち - エラーなし"
            Case "N"
                StatusString = "印字待ち/通過待ち - リボン/ラベルニアエンド"
            Case "O"
                StatusString = "印字待ち/通過待ち - バッファニアフル"
            Case "P"
                StatusString = "印字待ち/通過待ち - リボン/ラベルニアエンド＆バッファニアフル"
            Case "G"
                StatusString = "印字中 - エラーなし"
            Case "H"
                StatusString = "印字中 - リボン/ラベルニアエンド"
            Case "I"
                StatusString = "印字中 - バッファニアフル"
            Case "J"
                StatusString = "印字中 - リボン/ラベルニアエンド＆バッファニアフル"
            Case "o"
                StatusString = "印字吸着起動信号待ち/貼付中 - エラーなし"
            Case "p"
                StatusString = "印字吸着起動信号待ち/貼付中 - リボン/ラベルニアエンド"
            Case "q"
                StatusString = "印字吸着起動信号待ち/貼付中 - バッファニアフル"
            Case "r"
                StatusString = "印字吸着起動信号待ち/貼付中 - リボン/ラベルニアエンド＆バッファニアフル"
            Case "s"
                StatusString = "通過中 - エラーなし"
            Case "t"
                StatusString = "通過中 - リボン/ラベルニアエンド"
            Case "u"
                StatusString = "通過中 - バッファニアフル"
            Case "v"
                StatusString = "通過中 - リボン/ラベルニアエンド＆バッファニアフル"
            Case "S"
                StatusString = "解析・編集中 - エラーなし"
            Case "T"
                StatusString = "解析・編集中 - リボン/ラベルニアエンド"
            Case "U"
                StatusString = "解析・編集中 - バッファニアフル"
            Case "V"
                StatusString = "解析・編集中 - リボン/ラベルニアエンド＆バッファニアフル"
            Case "a"
                StatusString = "エラー - バッファオーバー"
            Case "b"
                StatusString = "エラー - ヘッドオープン"
            Case "c"
                StatusString = "エラー - ペーパーエンド/センサエラー/用紙詰まりエラー"
            Case "d"
                StatusString = "エラー - リボンエンド"
            Case "e"
                StatusString = "エラー - 印字飛びエラー/印字領域オーバー"
            Case "f"
                StatusString = "エラー - センサエラー/用紙詰まりエラー"
            Case "g"
                StatusString = "エラー - ヘッドエラー"
            Case "i"
                StatusString = "エラー - カードエラー"
            Case "k"
                StatusString = "エラー - その他のエラー"
            Case "l"
                StatusString = "エラー - ラベル貼付エラー"
            Case "#"
                StatusString = "エラー - ラベル吸着エラー"
            Case "$"
                StatusString = "エラー - 能力オーバー"
            Case "%"
                StatusString = "エラー - シリンダ動作不良"
            Case "+"
                StatusString = "エラー - ユーザー専用のエラー"
            Case Else
                StatusString = StatusChar
        End Select

        Return StatusString

    End Function


    Private Sub btnSelectLayout_Click(sender As Object, e As EventArgs) Handles btnSelectLayout.Click

        OpenFileDialog1.Title = "レイアウトファイルを選択"
        OpenFileDialog1.Filter = "レイアウトファイル (*.mllayx)|*.mllayx"
        OpenFileDialog1.FileName = ""

        If OpenFileDialog1.ShowDialog() = DialogResult.OK Then
            txtLayoutFile.Text = OpenFileDialog1.FileName
        End If

    End Sub


    Private Sub btnStop_Click(sender As Object, e As EventArgs) Handles btnStop.Click
        blnStop = True
    End Sub

    ''' <summary>
    ''' プリンタにキャンセルコマンドを送り、受信バッファを取り消します。
    ''' プリンタエラー等で発行を中止した場合は、キャンセルコマンドをプリンタに送信してください。
    ''' </summary>
    Private Sub btnSendCancel_Click(sender As Object, e As EventArgs) Handles btnSendCancel.Click

        Dim Result As Integer

        'キャンセルコマンド送信
        If ConnectPrinter() = True Then
            Result = MLComponent.SendCancel()

            If Result = 0 Then
                lblStatus.Text = "キャンセル送信成功"
            Else
                MessageBox.Show("キャンセルコマンド送信時にエラーが発生しました。(エラーコード : " & Result & ")")
            End If

            MLComponent.ClosePort()

        End If

    End Sub


    Private Sub btnSelectDataFile_Click(sender As Object, e As EventArgs) Handles btnSelectDataFile.Click

        OpenFileDialog1.Title = "データファイルを選択"
        OpenFileDialog1.Filter = "テキスト(タブ区切り) (*.txt)|*.txt|CSV(カンマ区切り) (*.csv)|*.csv|テキスト(スペース区切り) (*.prn)|*.prn|すべてのファイル (*.*)|*.*"
        OpenFileDialog1.FileName = ""

        If OpenFileDialog1.ShowDialog() = DialogResult.OK Then
            txtDataFile.Text = OpenFileDialog1.FileName
        End If

    End Sub

End Class
